<?php
/**
 * BIN Checker API - untuk cPanel
 * Usage: domain.com/bin?519234 atau domain.com/bin?bin=519234
 * Output: Bin :: 519234 MASTERCARD DEBIT ENHANCED FIRST FINANCIAL CREDIT UNION
 */
// --- PROXY (off = cuma pakai IP server, tidak pakai proxy) ---
$PROXY_URL = '';                    // proxy tetap (kosong = tidak pakai)
$PROXY_FALLBACK_ENABLED = false;   // false = off; true = kalau direct gagal coba list SOCKS5
$PROXY_LIST_URL = 'https://raw.githubusercontent.com/ClearProxy/checked-proxy-list/main/socks5/raw/all.txt';
$PROXY_CACHE_FILE = __DIR__ . '/.proxy_list_cache';
$PROXY_CACHE_TTL = 3600;
$PROXY_MAX_TRY = 20;
// --- Cache hasil BIN (kurangi kena limit) ---
$BIN_CACHE_DIR = __DIR__ . '/.bin_cache';
$BIN_CACHE_TTL = 86400;  // 24 jam
// --- Referensi: useragents.json (rotasi UA), webgl.json (fingerprint, bisa dipakai header nanti) ---
$USERAGENTS_FILE = __DIR__ . '/useragents.json';
$WEBGL_FILE = __DIR__ . '/webgl.json';
// --- akhir proxy ---

header('Content-Type: text/plain; charset=utf-8');
header('Access-Control-Allow-Origin: *');

/**
 * Ambil User-Agent + Accept-Language acak dari useragents.json agar request lebih alami, kurang kena blok.
 * Referensi: webgl.json (fingerprint GPU/browser) bisa dipakai nanti untuk header tambahan (mis. Sec-Ch-Ua).
 */
function get_random_ua_headers($useragentsPath) {
    static $uaList = null;
    $fallback = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/131.0.0.0 Safari/537.36';
    if ($uaList === null) {
        if (!is_file($useragentsPath)) return ['User-Agent' => $fallback];
        $raw = @file_get_contents($useragentsPath);
        $arr = @json_decode($raw, true);
        if (!is_array($arr)) return ['User-Agent' => $fallback];
        $uaList = [];
        foreach ($arr as $e) {
            if (!empty($e['userAgent']) && is_string($e['userAgent'])) {
                $uaList[] = [
                    'ua' => $e['userAgent'],
                    'lang' => isset($e['language']) ? $e['language'] : 'en-US,en;q=0.9',
                ];
            }
        }
    }
    if (empty($uaList)) return ['User-Agent' => $fallback];
    $e = $uaList[array_rand($uaList)];
    return ['User-Agent' => $e['ua'], 'Accept-Language' => $e['lang']];
}

function get_cached_bin_line($bin, $cacheDir, $ttl) {
    $f = rtrim($cacheDir, '/\\') . '/' . $bin . '.txt';
    if (!is_file($f)) return null;
    if (filemtime($f) + $ttl < time()) return null;
    $line = @file_get_contents($f);
    return ($line !== false && $line !== '') ? trim($line) : null;
}
function set_cached_bin_line($bin, $line, $cacheDir) {
    $dir = rtrim($cacheDir, '/\\');
    if (!is_dir($dir)) @mkdir($dir, 0755, true);
    @file_put_contents($dir . '/' . $bin . '.txt', $line, LOCK_EX);
}

/**
 * Ambil list proxy SOCKS5 (IP:port) dari cache atau dari URL, cache per TTL.
 * @return string[] array of 'ip:port'
 */
function get_socks5_proxy_list($listUrl, $cacheFile, $ttl) {
    $now = time();
    if (is_file($cacheFile)) {
        $raw = @file_get_contents($cacheFile);
        if ($raw !== false && preg_match('/^(\d+)\n(.*)/s', $raw, $m) && ($now - (int)$m[1]) < $ttl) {
            $lines = array_filter(array_map('trim', explode("\n", $m[2])));
            $list = [];
            foreach ($lines as $line) {
                if (preg_match('/^[\d.]+:\d+$/', $line)) {
                    $list[] = $line;
                }
            }
            if (!empty($list)) return $list;
        }
    }
    $ctx = stream_context_create(['http' => ['timeout' => 15]]);
    $body = @file_get_contents($listUrl, false, $ctx);
    if ($body === false) return [];
    $lines = array_filter(array_map('trim', explode("\n", $body)));
    $list = [];
    foreach ($lines as $line) {
        if (preg_match('/^[\d.]+:\d+$/', $line)) {
            $list[] = $line;
        }
    }
    @file_put_contents($cacheFile, $now . "\n" . implode("\n", $list), LOCK_EX);
    return $list;
}

/**
 * Fetch URL dengan cURL, optional proxy. User-Agent dirotasi dari useragents.json.
 */
function fetch_url($url, $proxyUrl = null, $useragentsPath = null) {
    $uaPath = $useragentsPath !== null ? $useragentsPath : __DIR__ . '/useragents.json';
    $headers = get_random_ua_headers($uaPath);
    $curlHeaders = ['User-Agent: ' . $headers['User-Agent']];
    if (!empty($headers['Accept-Language'])) $curlHeaders[] = 'Accept-Language: ' . $headers['Accept-Language'];
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT        => 12,
        CURLOPT_CONNECTTIMEOUT => 6,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_HTTPHEADER     => $curlHeaders,
    ]);
    if (!empty($proxyUrl)) {
        curl_setopt($ch, CURLOPT_PROXY, $proxyUrl);
        curl_setopt($ch, CURLOPT_PROXYTYPE, CURLPROXY_SOCKS5);
    }
    $html = curl_exec($ch);
    $err = curl_error($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    return ['body' => $html, 'error' => $err, 'http_code' => $code];
}

/** Cek apakah response HTML dari bincheck.io valid (ada konten BIN). */
function is_valid_bincheck_response($html) {
    if ($html === false || $html === '' || strlen($html) < 500) return false;
    return (stripos($html, 'Card Brand') !== false || stripos($html, 'BIN/IIN') !== false || stripos($html, 'valid BIN number') !== false);
}

// Ambil BIN dari query string: domain.com/bin?519234 atau domain.com/bin?bin=519234
$bin = null;
if (!empty($_GET['bin'])) {
    $bin = preg_replace('/\D/', '', (string)$_GET['bin']);
}
if (empty($bin) && !empty($_SERVER['QUERY_STRING'])) {
    // ?519234 -> QUERY_STRING = "519234"
    $qs = trim($_SERVER['QUERY_STRING']);
    $bin = preg_replace('/\D/', '', $qs);
}
if (empty($bin)) {
    $keys = array_keys($_GET);
    if (!empty($keys)) {
        $firstKey = $keys[0];
        $bin = preg_replace('/\D/', '', (string)$_GET[$firstKey]);
    }
}

if (empty($bin) || strlen($bin) < 4) {
    echo 'Error: BIN tidak valid. Gunakan: ?519234 atau ?bin=519234';
    exit;
}

// Ambil 6 digit pertama saja (standar BIN)
$bin = substr($bin, 0, 6);

// Pakai cache dulu (supaya tidak kena limit)
$cached = get_cached_bin_line($bin, $BIN_CACHE_DIR, $BIN_CACHE_TTL);
if ($cached !== null) {
    echo $cached;
    exit;
}

$url = 'https://bincheck.io/details/' . $bin;

if (!function_exists('curl_init')) {
    echo 'Error: cURL tidak tersedia di PHP.';
    exit;
}

$html = false;
$lastError = '';

// 1) Kalau $PROXY_URL di-set: langsung pakai proxy itu
if (!empty($PROXY_URL)) {
    $res = fetch_url($url, $PROXY_URL);
    $html = $res['body'];
    $lastError = $res['error'];
} else {
    // Coba direct ke bincheck.io (sampai 2x)
    for ($attempt = 0; $attempt < 2; $attempt++) {
        $res = fetch_url($url, null);
        $html = $res['body'];
        $lastError = $res['error'];
        if ($html !== false && is_valid_bincheck_response($html)) break;
        if ($attempt < 1) usleep(300000);
    }
    if ($PROXY_FALLBACK_ENABLED && ($html === false || $html === '' || !is_valid_bincheck_response($html))) {
        $blocked = ($res['http_code'] === 403 || $res['http_code'] === 429 || $res['http_code'] === 0);
        if ($blocked || $html === false || $html === '') {
            $list = get_socks5_proxy_list($PROXY_LIST_URL, $PROXY_CACHE_FILE, $PROXY_CACHE_TTL);
            $list = array_values(array_unique($list));
            shuffle($list);
            $tried = 0;
            foreach ($list as $entry) {
                if ($tried >= $PROXY_MAX_TRY) break;
                $res = fetch_url($url, 'socks5://' . $entry);
                $html = $res['body'];
                $lastError = $res['error'];
                if ($html !== false && is_valid_bincheck_response($html)) break;
                $tried++;
            }
        }
    }
}

// Fallback: binlist.net API (tanpa proxy)
if ($html === false || $html === '' || !is_valid_bincheck_response($html)) {
    $apiUrl = 'https://lookup.binlist.net/' . $bin;
    $uaHeaders = get_random_ua_headers($GLOBALS['USERAGENTS_FILE'] ?? __DIR__ . '/useragents.json');
    $binlistHeaders = ['Accept-Version: 3', 'User-Agent: ' . $uaHeaders['User-Agent']];
    if (!empty($uaHeaders['Accept-Language'])) $binlistHeaders[] = 'Accept-Language: ' . $uaHeaders['Accept-Language'];
    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => 8,
        CURLOPT_CONNECTTIMEOUT => 5,
        CURLOPT_HTTPHEADER     => $binlistHeaders,
    ]);
    $json = curl_exec($ch);
    $binlistCode = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    if ($json !== false) {
        $data = @json_decode($json, true);
        if (!empty($data['scheme']) || !empty($data['bank']['name'])) {
            $cardBrand = isset($data['scheme']) ? strtoupper($data['scheme']) : '';
            $cardType = isset($data['type']) ? strtoupper($data['type']) : '';
            $cardLevel = (isset($data['brand']) && stripos($data['brand'], 'enhanced') !== false) ? 'ENHANCED' : '';
            if (empty($cardLevel) && !empty($data['brand']) && stripos($data['brand'], 'standard') !== false) $cardLevel = 'STANDARD';
            $issuerName = isset($data['bank']['name']) ? trim($data['bank']['name']) : '';
            $parts = array_filter([$cardBrand, $cardType, $cardLevel, $issuerName]);
            $line = empty($parts) ? 'Bin            :: ' . $bin . ' (data tidak ditemukan)' : 'Bin            :: ' . $bin . ' ' . implode(' ', $parts);
            set_cached_bin_line($bin, $line, $BIN_CACHE_DIR);
            echo $line;
            exit;
        }
    }
    $rateLimitHint = ($res['http_code'] === 429 || $binlistCode === 429) ? ' Mungkin kena rate limit.' : '';
    echo 'Error: Tidak dapat mengambil data BIN. Coba lagi nanti.' . $rateLimitHint . ($lastError ? ' (' . $lastError . ')' : '');
    exit;
}

// Parse: BIN/IIN, Card Brand, Card Type, Card Level, Issuer Name / Bank
$binVal = $bin;
$cardBrand = '';
$cardType = '';
$cardLevel = '';
$issuerName = '';

// BIN/IIN - dari tabel atau meta
if (preg_match('/<td[^>]*>BIN\/IIN<\/td>\s*<td[^>]*>(\d+)<\/td>/s', $html, $m)) {
    $binVal = trim($m[1]);
}
if (preg_match('/Details for the BIN\/IIN:\s*<[^>]+>(\d+)<\/span>/', $html, $m)) {
    $binVal = trim($m[1]);
}

// Card Brand
if (preg_match('/<td[^>]*>Card Brand<\/td>\s*<td[^>]*>([^<]+)<\/td>/s', $html, $m)) {
    $cardBrand = trim(strip_tags($m[1]));
}

// Card Type
if (preg_match('/<td[^>]*>Card Type<\/td>\s*<td[^>]*>([^<]+)<\/td>/s', $html, $m)) {
    $cardType = trim(strip_tags($m[1]));
}

// Card Level
if (preg_match('/<td[^>]*>Card Level<\/td>\s*<td[^>]*>([^<]+)<\/td>/s', $html, $m)) {
    $cardLevel = trim(strip_tags($m[1]));
}

// Issuer Name / Bank (bisa ada link <a>)
if (preg_match('/<td[^>]*>Issuer Name \/ Bank<\/td>\s*<td[^>]*>(.*?)<\/td>/s', $html, $m)) {
    $issuerName = trim(strip_tags($m[1]));
}

// Fallback dari meta description jika tabel tidak ketemu
if (empty($cardBrand) && preg_match('/is a valid BIN number ([A-Z\s]+) issued by ([^.]+) in/i', $html, $m)) {
    if (empty($cardBrand)) $cardBrand = trim($m[1]);
    if (empty($issuerName)) $issuerName = trim($m[2]);
}

// Gabung: Bin :: 519234 MASTERCARD DEBIT ENHANCED FIRST FINANCIAL CREDIT UNION
$parts = array_filter([$cardBrand, $cardType, $cardLevel, $issuerName]);
$line = 'Bin            :: ' . $binVal . ' ' . implode(' ', $parts);

if (empty($parts)) {
    $line = 'Bin            :: ' . $binVal . ' (data tidak ditemukan)';
}

set_cached_bin_line($bin, $line, $BIN_CACHE_DIR);
echo $line;
